<?php
/**
 *
 * Users Email Stop. An extension for the phpBB Forum Software package.
 *
 * @copyright (c) 2025, Picaron, https://github.com/picaronin/
 * @license GNU General Public License, version 2 (GPL-2.0)
 *
 */

namespace pikaron\usersemailstop\event;

/**
 * Event listener
 */
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class listener implements EventSubscriberInterface
{
    /** @var \phpbb\request\request */
    protected $request;

    /** @var \phpbb\user */
    protected $user;

    /** @var \phpbb\auth\auth */
    protected $auth;

    /** @var \phpbb\template\template */
    protected $template;

    /** @var \phpbb\db\driver\driver_interface */
    protected $db;

    /** @var string */
    protected $phpbb_root_path;

    /** @var string */
    protected $phpEx;

    /** @var string */
    protected $table_prefix;

    /**
     * Listener constructor.
     *
     * @param \phpbb\request\request                             $request
     * @param \phpbb\user                                        $user
     * @param \phpbb\auth\auth                                   $auth
     * @param \phpbb\template\template                           $template
     * @param \phpbb\db\driver\driver_interface                  $db
     * @param string                                             $phpbb_root_path
     * @param string                                             $phpEx
     * @param string                                             $table_prefix
     *
     * @return void
     */
    public function __construct
    (
        \phpbb\request\request $request,
        \phpbb\user $user,
        \phpbb\auth\auth $auth,
        \phpbb\template\template $template,
        \phpbb\db\driver\driver_interface $db,
        $phpbb_root_path,
        $phpEx,
        $table_prefix
    )
    {
        $this->request               = $request;
        $this->user                  = $user;
        $this->auth                  = $auth;
        $this->template              = $template;
        $this->db                    = $db;
        $this->phpbb_root_path       = $phpbb_root_path;
        $this->phpEx                 = $phpEx;
        $this->table_prefix          = $table_prefix;
    }

    /**
    * Validate Users Email Stop
    *
    * @param \phpbb\event\data $event The event object
    * @return void
    * @access public
    */
    static public function getSubscribedEvents()
    {
        return array(
            'core.user_setup'                       => 'user_setup',
            'core.ucp_profile_reg_details_data'     => 'ucp_profile_reg_details_data',
            'core.ucp_profile_reg_details_sql_ary'  => 'ucp_profile_reg_details_sql_ary',
            'core.acp_users_overview_modify_data'   => 'acp_users_overview_modify_data',
            'core.acp_users_display_overview'       => 'acp_users_display_overview',
        );
    }

    /**
     * Setup User
     *
     * @param object $event
     *
     * @return void
     */
    public function user_setup($event)
    {
        $lang_set_ext = $event['lang_set_ext'];
        $lang_set_ext[] = [
            'ext_name'  => 'pikaron/usersemailstop',
            'lang_set'  => 'usersemailstop'
        ];
        $event['lang_set_ext'] = $lang_set_ext;

        // Force redirect to change Email -- Not for admins or moderators
        $isadmin = (isset($this->user->data['session_admin']) && $this->user->data['session_admin']) || $this->auth->acl_get('m_lock') ? true : false;
        if (!$isadmin && $this->user->data['users_email_stop'] && !empty($this->user->data['is_registered']))
        {
            if (strpos($this->user->page['query_string'], 'mode=reg_details') === false)
            {
                redirect(append_sid("{$this->phpbb_root_path}ucp.$this->phpEx", 'i=ucp_profile&amp;mode=reg_details'));
            }
        }
    }

    /**
     * Deactivate the email change condition
     *
     * @param object $event
     *
     * @return void
     */
    public function ucp_profile_reg_details_sql_ary($event)
    {
        $sql = $event['sql_ary'];

        // If email is changed, the condition is deactivated.
        if ($this->user->data['user_email'] != $sql['user_email'])
        {
            $sql['users_email_stop'] = 0;
            $event['sql_ary'] = $sql;

            // Restore User Notifications
            $this->user_restore_notifications($this->user->data['user_id']);
        }
    }

    /**
     * Show message in form.
     *
     * @return template
     */
    public function ucp_profile_reg_details_data($event)
    {
        if ($this->user->data['users_email_stop'])
        {
            if ($this->auth->acl_get('u_chgemail'))
            {
                $show = true;
                $noshow = false;
            }
            else
            {
                $show = false;
                $noshow = true;
            }
        }
        else
        {
            $show = false;
            $noshow = false;
        }

        $this->template->assign_vars(array(
            'USERSEMAILSTOP_FORCE_CHANGE_EMAIL' => $show,
            'USERSEMAILSTOP_NOT_FORCE_CHANGE_EMAIL' => $noshow
        ));
    }

    /**
     * Allow Administrator to block email account
     *
     * @param object $event The event object
     * @return null
     * @access public
     */
    public function acp_users_overview_modify_data($event)
    {
        // we capture the value of the variable usersemailstop
        $event['data'] = array_merge($event['data'], [
            'users_email_stop' => $this->request->variable('usersemailstop', 0),
        ]);

        // we include the value of the variable usersemailstop in sql_ary
        $event['sql_ary'] = array_merge($event['sql_ary'], [
            'users_email_stop' => $event['data']['users_email_stop'],
        ]);

        // If the value of the variable usersemailstop is yes
        if ($this->request->variable('usersemailstop', 0) == 1)
        {
            // We have removed any notification backups from this user.
            $sql = 'DELETE FROM ' . $this->table_prefix . 'users_email_stop' . '
                WHERE user_id = ' . (int) $event['user_row']['user_id'];
            $this->db->sql_query($sql);

            // We get all user-defined email notifications.
            $sql = 'SELECT item_type
                    FROM ' . USER_NOTIFICATIONS_TABLE . '
                    WHERE user_id = ' . (int) $event['user_row']['user_id'] . "
                    AND method = 'notification.method.email' AND notify = 1";

            $result = $this->db->sql_query($sql);

            while ($row = $this->db->sql_fetchrow($result))
            {
                // We backup all user-defined email notifications.
                $this->db->sql_query('INSERT INTO ' . $this->table_prefix . 'users_email_stop' . ' ' . $this->db->sql_build_array('INSERT', array('item_type' => $row['item_type'], 'user_id'=> (int) $event['user_row']['user_id'])));
            }

            $this->db->sql_freeresult($result);

            // We delete all user-defined email notifications in the USER_NOTIFICATIONS_TABLE.
            $sql = 'UPDATE ' . USER_NOTIFICATIONS_TABLE . " SET notify = '0'
                    WHERE method = 'notification.method.email'
                    AND user_id =" . (int) $event['user_row']['user_id'];
            $this->db->sql_query($sql);

        } else {

            // Restore User Notifications
            $this->user_restore_notifications($event['user_row']['user_id']);
        }

    }

    /**
     * Display Overview in ACP
     *
     * @param object $event The event object
     * @return null
     * @access public
     */
    public function acp_users_display_overview($event)
    {
        $this->template->assign_vars([
            'USERSEMAILSTOP_ENABLED' => $event['user_row']['users_email_stop'] == 1 ? true : false,
        ]);
    }

    /**
     * Restore User Notifications
     *
     * @param object $event The event object
     * @return null
     * @access public
     */
    public function user_restore_notifications($data)
    {
        // We get all backup user-defined email notifications.
        $rows = array();
        $sql = 'SELECT user_id, item_type
                FROM ' . $this->table_prefix . 'users_email_stop' . '
                WHERE user_id = ' . (int) $data;
        $result = $this->db->sql_query($sql);
        while ($row = $this->db->sql_fetchrow($result))
        {
            $rows[] = $row;
        }
        $this->db->sql_freeresult($result);

        if (count($rows) > 0)
        {
            foreach ($rows as $key => $type)
            {
                // We restore all user-defined email notifications in the backup users_email_stop table.
                $sql = 'UPDATE ' . USER_NOTIFICATIONS_TABLE . " SET notify = '1'
                        WHERE item_type = '" . $type['item_type'] . "' AND method = 'notification.method.email' AND user_id =" . $type['user_id'];
                $this->db->sql_query($sql);
            }

            // We have removed any notification backups from this user.
            $sql = 'DELETE FROM ' . $this->table_prefix . 'users_email_stop' . '
                WHERE user_id = ' . (int) $data;
            $this->db->sql_query($sql);
        }
    }

}
